/*
Copyright the Velero contributors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

// This code embeds the CRD manifests in ../bases in ../crds/crds.go

package main

import (
	"bytes"
	"compress/gzip"
	"fmt"
	"io"
	"log"
	"os"
	"text/template"
)

// This is relative to config/crd/crds
const goHeaderFile = "../../../../hack/boilerplate.go.txt"

const tpl = `{{.GoHeader}}
// Code generated by crds_generate.go; DO NOT EDIT.

package crds

import (
	"bytes"
	"compress/gzip"
	"io"

	apiextinstall "k8s.io/apiextensions-apiserver/pkg/apis/apiextensions/install"
	apiextv1 "k8s.io/apiextensions-apiserver/pkg/apis/apiextensions/v1"
	"k8s.io/client-go/kubernetes/scheme"
)

var rawCRDs = [][]byte{
{{- range .RawCRDs }}
	[]byte({{ . }}),
{{- end }}
}

var CRDs = crds()

func crds() []*apiextv1.CustomResourceDefinition {
	apiextinstall.Install(scheme.Scheme)
	decode := scheme.Codecs.UniversalDeserializer().Decode
	var objs []*apiextv1.CustomResourceDefinition
	for _, crd := range rawCRDs {
		gzr, err := gzip.NewReader(bytes.NewReader(crd))
		if err != nil {
			panic(err)
		}
		bytes, err := io.ReadAll(gzr)
		if err != nil {
			panic(err)
		}
		gzr.Close()

		obj, _, err := decode(bytes, nil, nil)
		if err != nil {
			panic(err)
		}
		objs = append(objs, obj.(*apiextv1.CustomResourceDefinition))
	}
	return objs
}
`

type templateData struct {
	GoHeader string
	RawCRDs  []string
}

func main() {
	headerBytes, err := os.ReadFile(goHeaderFile)
	if err != nil {
		log.Fatalln(err)
	}

	data := templateData{
		GoHeader: string(headerBytes),
	}

	// This is relative to config/crd/crds
	manifests, err := os.ReadDir("../bases")
	if err != nil {
		log.Fatalln(err)
	}

	for _, crd := range manifests {
		file, err := os.Open("../bases/" + crd.Name())
		if err != nil {
			log.Fatalln(err)
		}

		// gzip compress manifest
		var buf bytes.Buffer
		gzw := gzip.NewWriter(&buf)
		if _, err := io.Copy(gzw, file); err != nil {
			log.Fatalln(err)
		}
		file.Close()
		gzw.Close()

		data.RawCRDs = append(data.RawCRDs, fmt.Sprintf("%q", buf.Bytes()))
	}

	t, err := template.New("crd").Parse(tpl)
	if err != nil {
		log.Fatalln(err)
	}

	out, err := os.Create("crds.go")
	if err != nil {
		log.Fatalln(err)
	}

	if err := t.Execute(out, data); err != nil {
		log.Fatalln(err)
	}
}
