/*
Copyright 2022 The Tekton Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package v1

import (
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// Step runs a subcomponent of a Task
type Step struct {
	// Name of the Step specified as a DNS_LABEL.
	// Each Step in a Task must have a unique name.
	Name string `json:"name" protobuf:"bytes,1,opt,name=name"`
	// Docker image name.
	// More info: https://kubernetes.io/docs/concepts/containers/images
	// +optional
	Image string `json:"image,omitempty" protobuf:"bytes,2,opt,name=image"`
	// Entrypoint array. Not executed within a shell.
	// The image's ENTRYPOINT is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the container's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Command []string `json:"command,omitempty" protobuf:"bytes,3,rep,name=command"`
	// Arguments to the entrypoint.
	// The image's CMD is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the container's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Args []string `json:"args,omitempty" protobuf:"bytes,4,rep,name=args"`
	// Step's working directory.
	// If not specified, the container runtime's default will be used, which
	// might be configured in the container image.
	// Cannot be updated.
	// +optional
	WorkingDir string `json:"workingDir,omitempty" protobuf:"bytes,5,opt,name=workingDir"`
	// List of sources to populate environment variables in the Step.
	// The keys defined within a source must be a C_IDENTIFIER. All invalid keys
	// will be reported as an event when the Step is starting. When a key exists in multiple
	// sources, the value associated with the last source will take precedence.
	// Values defined by an Env with a duplicate key will take precedence.
	// Cannot be updated.
	// +optional
	// +listType=atomic
	EnvFrom []corev1.EnvFromSource `json:"envFrom,omitempty" protobuf:"bytes,19,rep,name=envFrom"`
	// List of environment variables to set in the Step.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=name
	// +patchStrategy=merge
	// +listType=atomic
	Env []corev1.EnvVar `json:"env,omitempty" patchMergeKey:"name" patchStrategy:"merge" protobuf:"bytes,7,rep,name=env"`
	// ComputeResources required by this Step.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
	// +optional
	ComputeResources corev1.ResourceRequirements `json:"computeResources,omitempty" protobuf:"bytes,8,opt,name=computeResources"`
	// Volumes to mount into the Step's filesystem.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=mountPath
	// +patchStrategy=merge
	// +listType=atomic
	VolumeMounts []corev1.VolumeMount `json:"volumeMounts,omitempty" patchMergeKey:"mountPath" patchStrategy:"merge" protobuf:"bytes,9,rep,name=volumeMounts"`
	// volumeDevices is the list of block devices to be used by the Step.
	// +patchMergeKey=devicePath
	// +patchStrategy=merge
	// +optional
	// +listType=atomic
	VolumeDevices []corev1.VolumeDevice `json:"volumeDevices,omitempty" patchMergeKey:"devicePath" patchStrategy:"merge" protobuf:"bytes,21,rep,name=volumeDevices"`
	// Image pull policy.
	// One of Always, Never, IfNotPresent.
	// Defaults to Always if :latest tag is specified, or IfNotPresent otherwise.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/containers/images#updating-images
	// +optional
	ImagePullPolicy corev1.PullPolicy `json:"imagePullPolicy,omitempty" protobuf:"bytes,14,opt,name=imagePullPolicy,casttype=PullPolicy"`
	// SecurityContext defines the security options the Step should be run with.
	// If set, the fields of SecurityContext override the equivalent fields of PodSecurityContext.
	// More info: https://kubernetes.io/docs/tasks/configure-pod-container/security-context/
	// +optional
	SecurityContext *corev1.SecurityContext `json:"securityContext,omitempty" protobuf:"bytes,15,opt,name=securityContext"`

	// Script is the contents of an executable file to execute.
	//
	// If Script is not empty, the Step cannot have an Command and the Args will be passed to the Script.
	// +optional
	Script string `json:"script,omitempty"`

	// Timeout is the time after which the step times out. Defaults to never.
	// Refer to Go's ParseDuration documentation for expected format: https://golang.org/pkg/time/#ParseDuration
	// +optional
	Timeout *metav1.Duration `json:"timeout,omitempty"`

	// This is an alpha field. You must set the "enable-api-fields" feature flag to "alpha"
	// for this field to be supported.
	//
	// Workspaces is a list of workspaces from the Task that this Step wants
	// exclusive access to. Adding a workspace to this list means that any
	// other Step or Sidecar that does not also request this Workspace will
	// not have access to it.
	// +optional
	// +listType=atomic
	Workspaces []WorkspaceUsage `json:"workspaces,omitempty"`

	// OnError defines the exiting behavior of a container on error
	// can be set to [ continue | stopAndFail ]
	OnError OnErrorType `json:"onError,omitempty"`
	// Stores configuration for the stdout stream of the step.
	// +optional
	StdoutConfig *StepOutputConfig `json:"stdoutConfig,omitempty"`
	// Stores configuration for the stderr stream of the step.
	// +optional
	StderrConfig *StepOutputConfig `json:"stderrConfig,omitempty"`
	// Contains the reference to an existing StepAction.
	//+optional
	Ref *Ref `json:"ref,omitempty"`
	// Params declares parameters passed to this step action.
	// +optional
	// +listType=atomic
	Params Params `json:"params,omitempty"`
	// Results declares StepResults produced by the Step.
	//
	// It can be used in an inlined Step when used to store Results to $(step.results.resultName.path).
	// It cannot be used when referencing StepActions using [v1.Step.Ref].
	// The Results declared by the StepActions will be stored here instead.
	// +optional
	// +listType=atomic
	Results []StepResult `json:"results,omitempty"`

	// When is a list of when expressions that need to be true for the task to run
	// +optional
	When StepWhenExpressions `json:"when,omitempty"`
}

// Ref can be used to refer to a specific instance of a StepAction.
type Ref struct {
	// Name of the referenced step
	Name string `json:"name,omitempty"`
	// ResolverRef allows referencing a StepAction in a remote location
	// like a git repo.
	// +optional
	ResolverRef `json:",omitempty"`
}

// OnErrorType defines a list of supported exiting behavior of a container on error
type OnErrorType string

const (
	// StopAndFail indicates exit the taskRun if the container exits with non-zero exit code
	StopAndFail OnErrorType = "stopAndFail"
	// Continue indicates continue executing the rest of the steps irrespective of the container exit code
	Continue OnErrorType = "continue"
)

// StepOutputConfig stores configuration for a step output stream.
type StepOutputConfig struct {
	// Path to duplicate stdout stream to on container's local filesystem.
	// +optional
	Path string `json:"path,omitempty"`
}

// ToK8sContainer converts the Step to a Kubernetes Container struct
func (s *Step) ToK8sContainer() *corev1.Container {
	return &corev1.Container{
		Name:            s.Name,
		Image:           s.Image,
		Command:         s.Command,
		Args:            s.Args,
		WorkingDir:      s.WorkingDir,
		EnvFrom:         s.EnvFrom,
		Env:             s.Env,
		Resources:       s.ComputeResources,
		VolumeMounts:    s.VolumeMounts,
		VolumeDevices:   s.VolumeDevices,
		ImagePullPolicy: s.ImagePullPolicy,
		SecurityContext: s.SecurityContext,
	}
}

// SetContainerFields sets the fields of the Step to the values of the corresponding fields in the Container
func (s *Step) SetContainerFields(c corev1.Container) {
	s.Name = c.Name
	s.Image = c.Image
	s.Command = c.Command
	s.Args = c.Args
	s.WorkingDir = c.WorkingDir
	s.EnvFrom = c.EnvFrom
	s.Env = c.Env
	s.ComputeResources = c.Resources
	s.VolumeMounts = c.VolumeMounts
	s.VolumeDevices = c.VolumeDevices
	s.ImagePullPolicy = c.ImagePullPolicy
	s.SecurityContext = c.SecurityContext
}

// GetVarSubstitutionExpressions walks all the places a substitution reference can be used
func (s *Step) GetVarSubstitutionExpressions() []string {
	var allExpressions []string
	allExpressions = append(allExpressions, validateString(s.Name)...)
	allExpressions = append(allExpressions, validateString(s.Image)...)
	allExpressions = append(allExpressions, validateString(string(s.ImagePullPolicy))...)
	allExpressions = append(allExpressions, validateString(s.Script)...)
	allExpressions = append(allExpressions, validateString(s.WorkingDir)...)
	for _, cmd := range s.Command {
		allExpressions = append(allExpressions, validateString(cmd)...)
	}
	for _, arg := range s.Args {
		allExpressions = append(allExpressions, validateString(arg)...)
	}
	for _, env := range s.Env {
		allExpressions = append(allExpressions, validateString(env.Value)...)
		if env.ValueFrom != nil {
			if env.ValueFrom.SecretKeyRef != nil {
				allExpressions = append(allExpressions, validateString(env.ValueFrom.SecretKeyRef.Key)...)
				allExpressions = append(allExpressions, validateString(env.ValueFrom.SecretKeyRef.LocalObjectReference.Name)...)
			}
			if env.ValueFrom.ConfigMapKeyRef != nil {
				allExpressions = append(allExpressions, validateString(env.ValueFrom.ConfigMapKeyRef.Key)...)
				allExpressions = append(allExpressions, validateString(env.ValueFrom.ConfigMapKeyRef.LocalObjectReference.Name)...)
			}
		}
	}
	return allExpressions
}

// StepTemplate is a template for a Step
type StepTemplate struct {
	// Image reference name.
	// More info: https://kubernetes.io/docs/concepts/containers/images
	// +optional
	Image string `json:"image,omitempty" protobuf:"bytes,2,opt,name=image"`
	// Entrypoint array. Not executed within a shell.
	// The image's ENTRYPOINT is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the Step's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Command []string `json:"command,omitempty" protobuf:"bytes,3,rep,name=command"`
	// Arguments to the entrypoint.
	// The image's CMD is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the Step's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Args []string `json:"args,omitempty" protobuf:"bytes,4,rep,name=args"`
	// Step's working directory.
	// If not specified, the container runtime's default will be used, which
	// might be configured in the container image.
	// Cannot be updated.
	// +optional
	WorkingDir string `json:"workingDir,omitempty" protobuf:"bytes,5,opt,name=workingDir"`
	// List of sources to populate environment variables in the Step.
	// The keys defined within a source must be a C_IDENTIFIER. All invalid keys
	// will be reported as an event when the Step is starting. When a key exists in multiple
	// sources, the value associated with the last source will take precedence.
	// Values defined by an Env with a duplicate key will take precedence.
	// Cannot be updated.
	// +optional
	// +listType=atomic
	EnvFrom []corev1.EnvFromSource `json:"envFrom,omitempty" protobuf:"bytes,19,rep,name=envFrom"`
	// List of environment variables to set in the Step.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=name
	// +patchStrategy=merge
	// +listType=atomic
	Env []corev1.EnvVar `json:"env,omitempty" patchMergeKey:"name" patchStrategy:"merge" protobuf:"bytes,7,rep,name=env"`
	// ComputeResources required by this Step.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
	// +optional
	ComputeResources corev1.ResourceRequirements `json:"computeResources,omitempty" protobuf:"bytes,8,opt,name=computeResources"`
	// Volumes to mount into the Step's filesystem.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=mountPath
	// +patchStrategy=merge
	// +listType=atomic
	VolumeMounts []corev1.VolumeMount `json:"volumeMounts,omitempty" patchMergeKey:"mountPath" patchStrategy:"merge" protobuf:"bytes,9,rep,name=volumeMounts"`
	// volumeDevices is the list of block devices to be used by the Step.
	// +patchMergeKey=devicePath
	// +patchStrategy=merge
	// +optional
	// +listType=atomic
	VolumeDevices []corev1.VolumeDevice `json:"volumeDevices,omitempty" patchMergeKey:"devicePath" patchStrategy:"merge" protobuf:"bytes,21,rep,name=volumeDevices"`
	// Image pull policy.
	// One of Always, Never, IfNotPresent.
	// Defaults to Always if :latest tag is specified, or IfNotPresent otherwise.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/containers/images#updating-images
	// +optional
	ImagePullPolicy corev1.PullPolicy `json:"imagePullPolicy,omitempty" protobuf:"bytes,14,opt,name=imagePullPolicy,casttype=PullPolicy"`
	// SecurityContext defines the security options the Step should be run with.
	// If set, the fields of SecurityContext override the equivalent fields of PodSecurityContext.
	// More info: https://kubernetes.io/docs/tasks/configure-pod-container/security-context/
	// +optional
	SecurityContext *corev1.SecurityContext `json:"securityContext,omitempty" protobuf:"bytes,15,opt,name=securityContext"`
}

// SetContainerFields sets the fields of the Step to the values of the corresponding fields in the Container
func (s *StepTemplate) SetContainerFields(c corev1.Container) {
	s.Image = c.Image
	s.Command = c.Command
	s.Args = c.Args
	s.WorkingDir = c.WorkingDir
	s.EnvFrom = c.EnvFrom
	s.Env = c.Env
	s.ComputeResources = c.Resources
	s.VolumeMounts = c.VolumeMounts
	s.VolumeDevices = c.VolumeDevices
	s.ImagePullPolicy = c.ImagePullPolicy
	s.SecurityContext = c.SecurityContext
}

// ToK8sContainer converts the StepTemplate to a Kubernetes Container struct
func (s *StepTemplate) ToK8sContainer() *corev1.Container {
	return &corev1.Container{
		Image:           s.Image,
		Command:         s.Command,
		Args:            s.Args,
		WorkingDir:      s.WorkingDir,
		EnvFrom:         s.EnvFrom,
		Env:             s.Env,
		Resources:       s.ComputeResources,
		VolumeMounts:    s.VolumeMounts,
		VolumeDevices:   s.VolumeDevices,
		ImagePullPolicy: s.ImagePullPolicy,
		SecurityContext: s.SecurityContext,
	}
}

// Sidecar has nearly the same data structure as Step but does not have the ability to timeout.
type Sidecar struct {
	// Name of the Sidecar specified as a DNS_LABEL.
	// Each Sidecar in a Task must have a unique name (DNS_LABEL).
	// Cannot be updated.
	Name string `json:"name" protobuf:"bytes,1,opt,name=name"`
	// Image reference name.
	// More info: https://kubernetes.io/docs/concepts/containers/images
	// +optional
	Image string `json:"image,omitempty" protobuf:"bytes,2,opt,name=image"`
	// Entrypoint array. Not executed within a shell.
	// The image's ENTRYPOINT is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the Sidecar's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Command []string `json:"command,omitempty" protobuf:"bytes,3,rep,name=command"`
	// Arguments to the entrypoint.
	// The image's CMD is used if this is not provided.
	// Variable references $(VAR_NAME) are expanded using the Sidecar's environment. If a variable
	// cannot be resolved, the reference in the input string will be unchanged. Double $$ are reduced
	// to a single $, which allows for escaping the $(VAR_NAME) syntax: i.e. "$$(VAR_NAME)" will
	// produce the string literal "$(VAR_NAME)". Escaped references will never be expanded, regardless
	// of whether the variable exists or not. Cannot be updated.
	// More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
	// +optional
	// +listType=atomic
	Args []string `json:"args,omitempty" protobuf:"bytes,4,rep,name=args"`
	// Sidecar's working directory.
	// If not specified, the container runtime's default will be used, which
	// might be configured in the container image.
	// Cannot be updated.
	// +optional
	WorkingDir string `json:"workingDir,omitempty" protobuf:"bytes,5,opt,name=workingDir"`
	// List of ports to expose from the Sidecar. Exposing a port here gives
	// the system additional information about the network connections a
	// container uses, but is primarily informational. Not specifying a port here
	// DOES NOT prevent that port from being exposed. Any port which is
	// listening on the default "0.0.0.0" address inside a container will be
	// accessible from the network.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=containerPort
	// +patchStrategy=merge
	// +listType=map
	// +listMapKey=containerPort
	// +listMapKey=protocol
	Ports []corev1.ContainerPort `json:"ports,omitempty" patchMergeKey:"containerPort" patchStrategy:"merge" protobuf:"bytes,6,rep,name=ports"`
	// List of sources to populate environment variables in the Sidecar.
	// The keys defined within a source must be a C_IDENTIFIER. All invalid keys
	// will be reported as an event when the container is starting. When a key exists in multiple
	// sources, the value associated with the last source will take precedence.
	// Values defined by an Env with a duplicate key will take precedence.
	// Cannot be updated.
	// +optional
	// +listType=atomic
	EnvFrom []corev1.EnvFromSource `json:"envFrom,omitempty" protobuf:"bytes,19,rep,name=envFrom"`
	// List of environment variables to set in the Sidecar.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=name
	// +patchStrategy=merge
	// +listType=atomic
	Env []corev1.EnvVar `json:"env,omitempty" patchMergeKey:"name" patchStrategy:"merge" protobuf:"bytes,7,rep,name=env"`
	// ComputeResources required by this Sidecar.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
	// +optional
	ComputeResources corev1.ResourceRequirements `json:"computeResources,omitempty" protobuf:"bytes,8,opt,name=computeResources"`
	// Volumes to mount into the Sidecar's filesystem.
	// Cannot be updated.
	// +optional
	// +patchMergeKey=mountPath
	// +patchStrategy=merge
	// +listType=atomic
	VolumeMounts []corev1.VolumeMount `json:"volumeMounts,omitempty" patchMergeKey:"mountPath" patchStrategy:"merge" protobuf:"bytes,9,rep,name=volumeMounts"`
	// volumeDevices is the list of block devices to be used by the Sidecar.
	// +patchMergeKey=devicePath
	// +patchStrategy=merge
	// +optional
	// +listType=atomic
	VolumeDevices []corev1.VolumeDevice `json:"volumeDevices,omitempty" patchMergeKey:"devicePath" patchStrategy:"merge" protobuf:"bytes,21,rep,name=volumeDevices"`
	// Periodic probe of Sidecar liveness.
	// Container will be restarted if the probe fails.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle#container-probes
	// +optional
	LivenessProbe *corev1.Probe `json:"livenessProbe,omitempty" protobuf:"bytes,10,opt,name=livenessProbe"`
	// Periodic probe of Sidecar service readiness.
	// Container will be removed from service endpoints if the probe fails.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle#container-probes
	// +optional
	ReadinessProbe *corev1.Probe `json:"readinessProbe,omitempty" protobuf:"bytes,11,opt,name=readinessProbe"`
	// StartupProbe indicates that the Pod the Sidecar is running in has successfully initialized.
	// If specified, no other probes are executed until this completes successfully.
	// If this probe fails, the Pod will be restarted, just as if the livenessProbe failed.
	// This can be used to provide different probe parameters at the beginning of a Pod's lifecycle,
	// when it might take a long time to load data or warm a cache, than during steady-state operation.
	// This cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle#container-probes
	// +optional
	StartupProbe *corev1.Probe `json:"startupProbe,omitempty" protobuf:"bytes,22,opt,name=startupProbe"`
	// Actions that the management system should take in response to Sidecar lifecycle events.
	// Cannot be updated.
	// +optional
	Lifecycle *corev1.Lifecycle `json:"lifecycle,omitempty" protobuf:"bytes,12,opt,name=lifecycle"`
	// Optional: Path at which the file to which the Sidecar's termination message
	// will be written is mounted into the Sidecar's filesystem.
	// Message written is intended to be brief final status, such as an assertion failure message.
	// Will be truncated by the node if greater than 4096 bytes. The total message length across
	// all containers will be limited to 12kb.
	// Defaults to /dev/termination-log.
	// Cannot be updated.
	// +optional
	TerminationMessagePath string `json:"terminationMessagePath,omitempty" protobuf:"bytes,13,opt,name=terminationMessagePath"`
	// Indicate how the termination message should be populated. File will use the contents of
	// terminationMessagePath to populate the Sidecar status message on both success and failure.
	// FallbackToLogsOnError will use the last chunk of Sidecar log output if the termination
	// message file is empty and the Sidecar exited with an error.
	// The log output is limited to 2048 bytes or 80 lines, whichever is smaller.
	// Defaults to File.
	// Cannot be updated.
	// +optional
	TerminationMessagePolicy corev1.TerminationMessagePolicy `json:"terminationMessagePolicy,omitempty" protobuf:"bytes,20,opt,name=terminationMessagePolicy,casttype=TerminationMessagePolicy"`
	// Image pull policy.
	// One of Always, Never, IfNotPresent.
	// Defaults to Always if :latest tag is specified, or IfNotPresent otherwise.
	// Cannot be updated.
	// More info: https://kubernetes.io/docs/concepts/containers/images#updating-images
	// +optional
	ImagePullPolicy corev1.PullPolicy `json:"imagePullPolicy,omitempty" protobuf:"bytes,14,opt,name=imagePullPolicy,casttype=PullPolicy"`
	// SecurityContext defines the security options the Sidecar should be run with.
	// If set, the fields of SecurityContext override the equivalent fields of PodSecurityContext.
	// More info: https://kubernetes.io/docs/tasks/configure-pod-container/security-context/
	// +optional
	SecurityContext *corev1.SecurityContext `json:"securityContext,omitempty" protobuf:"bytes,15,opt,name=securityContext"`

	// Variables for interactive containers, these have very specialized use-cases (e.g. debugging)
	// and shouldn't be used for general purpose containers.

	// Whether this Sidecar should allocate a buffer for stdin in the container runtime. If this
	// is not set, reads from stdin in the Sidecar will always result in EOF.
	// Default is false.
	// +optional
	Stdin bool `json:"stdin,omitempty" protobuf:"varint,16,opt,name=stdin"`
	// Whether the container runtime should close the stdin channel after it has been opened by
	// a single attach. When stdin is true the stdin stream will remain open across multiple attach
	// sessions. If stdinOnce is set to true, stdin is opened on Sidecar start, is empty until the
	// first client attaches to stdin, and then remains open and accepts data until the client disconnects,
	// at which time stdin is closed and remains closed until the Sidecar is restarted. If this
	// flag is false, a container processes that reads from stdin will never receive an EOF.
	// Default is false
	// +optional
	StdinOnce bool `json:"stdinOnce,omitempty" protobuf:"varint,17,opt,name=stdinOnce"`
	// Whether this Sidecar should allocate a TTY for itself, also requires 'stdin' to be true.
	// Default is false.
	// +optional
	TTY bool `json:"tty,omitempty" protobuf:"varint,18,opt,name=tty"`

	// Script is the contents of an executable file to execute.
	//
	// If Script is not empty, the Step cannot have an Command or Args.
	// +optional
	Script string `json:"script,omitempty"`

	// This is an alpha field. You must set the "enable-api-fields" feature flag to "alpha"
	// for this field to be supported.
	//
	// Workspaces is a list of workspaces from the Task that this Sidecar wants
	// exclusive access to. Adding a workspace to this list means that any
	// other Step or Sidecar that does not also request this Workspace will
	// not have access to it.
	// +optional
	// +listType=atomic
	Workspaces []WorkspaceUsage `json:"workspaces,omitempty"`

	// RestartPolicy refers to kubernetes RestartPolicy. It can only be set for an
	// initContainer and must have it's policy set to "Always". It is currently
	// left optional to help support Kubernetes versions prior to 1.29 when this feature
	// was introduced.
	// +optional
	RestartPolicy *corev1.ContainerRestartPolicy `json:"restartPolicy,omitempty"`
}

// ToK8sContainer converts the Sidecar to a Kubernetes Container struct
func (s *Sidecar) ToK8sContainer() *corev1.Container {
	if s.RestartPolicy == nil {
		return &corev1.Container{
			Name:                     s.Name,
			Image:                    s.Image,
			Command:                  s.Command,
			Args:                     s.Args,
			WorkingDir:               s.WorkingDir,
			Ports:                    s.Ports,
			EnvFrom:                  s.EnvFrom,
			Env:                      s.Env,
			Resources:                s.ComputeResources,
			VolumeMounts:             s.VolumeMounts,
			VolumeDevices:            s.VolumeDevices,
			LivenessProbe:            s.LivenessProbe,
			ReadinessProbe:           s.ReadinessProbe,
			StartupProbe:             s.StartupProbe,
			Lifecycle:                s.Lifecycle,
			TerminationMessagePath:   s.TerminationMessagePath,
			TerminationMessagePolicy: s.TerminationMessagePolicy,
			ImagePullPolicy:          s.ImagePullPolicy,
			SecurityContext:          s.SecurityContext,
			Stdin:                    s.Stdin,
			StdinOnce:                s.StdinOnce,
			TTY:                      s.TTY,
		}
	}
	return &corev1.Container{
		Name:                     s.Name,
		Image:                    s.Image,
		Command:                  s.Command,
		Args:                     s.Args,
		WorkingDir:               s.WorkingDir,
		Ports:                    s.Ports,
		EnvFrom:                  s.EnvFrom,
		Env:                      s.Env,
		Resources:                s.ComputeResources,
		VolumeMounts:             s.VolumeMounts,
		VolumeDevices:            s.VolumeDevices,
		LivenessProbe:            s.LivenessProbe,
		ReadinessProbe:           s.ReadinessProbe,
		RestartPolicy:            s.RestartPolicy,
		StartupProbe:             s.StartupProbe,
		Lifecycle:                s.Lifecycle,
		TerminationMessagePath:   s.TerminationMessagePath,
		TerminationMessagePolicy: s.TerminationMessagePolicy,
		ImagePullPolicy:          s.ImagePullPolicy,
		SecurityContext:          s.SecurityContext,
		Stdin:                    s.Stdin,
		StdinOnce:                s.StdinOnce,
		TTY:                      s.TTY,
	}
}

// SetContainerFields sets the fields of the Sidecar to the values of the corresponding fields in the Container
func (s *Sidecar) SetContainerFields(c corev1.Container) {
	s.Name = c.Name
	s.Image = c.Image
	s.Command = c.Command
	s.Args = c.Args
	s.WorkingDir = c.WorkingDir
	s.Ports = c.Ports
	s.EnvFrom = c.EnvFrom
	s.Env = c.Env
	s.ComputeResources = c.Resources
	s.VolumeMounts = c.VolumeMounts
	s.VolumeDevices = c.VolumeDevices
	s.LivenessProbe = c.LivenessProbe
	s.ReadinessProbe = c.ReadinessProbe
	s.StartupProbe = c.StartupProbe
	s.Lifecycle = c.Lifecycle
	s.TerminationMessagePath = c.TerminationMessagePath
	s.TerminationMessagePolicy = c.TerminationMessagePolicy
	s.ImagePullPolicy = c.ImagePullPolicy
	s.SecurityContext = c.SecurityContext
	s.Stdin = c.Stdin
	s.StdinOnce = c.StdinOnce
	s.TTY = c.TTY
	s.RestartPolicy = c.RestartPolicy
}

// GetVarSubstitutionExpressions walks all the places a substitution reference can be used
func (s *Sidecar) GetVarSubstitutionExpressions() []string {
	var allExpressions []string
	allExpressions = append(allExpressions, validateString(s.Name)...)
	allExpressions = append(allExpressions, validateString(s.Image)...)
	allExpressions = append(allExpressions, validateString(string(s.ImagePullPolicy))...)
	allExpressions = append(allExpressions, validateString(s.Script)...)
	allExpressions = append(allExpressions, validateString(s.WorkingDir)...)
	for _, cmd := range s.Command {
		allExpressions = append(allExpressions, validateString(cmd)...)
	}
	for _, arg := range s.Args {
		allExpressions = append(allExpressions, validateString(arg)...)
	}
	for _, env := range s.Env {
		allExpressions = append(allExpressions, validateString(env.Value)...)
		if env.ValueFrom != nil {
			if env.ValueFrom.SecretKeyRef != nil {
				allExpressions = append(allExpressions, validateString(env.ValueFrom.SecretKeyRef.Key)...)
				allExpressions = append(allExpressions, validateString(env.ValueFrom.SecretKeyRef.LocalObjectReference.Name)...)
			}
			if env.ValueFrom.ConfigMapKeyRef != nil {
				allExpressions = append(allExpressions, validateString(env.ValueFrom.ConfigMapKeyRef.Key)...)
				allExpressions = append(allExpressions, validateString(env.ValueFrom.ConfigMapKeyRef.LocalObjectReference.Name)...)
			}
		}
	}
	return allExpressions
}
